/*
 * Decompiled with CFR 0.152.
 */
package net.shibboleth.oidc.security.impl;

import com.google.common.base.Strings;
import com.nimbusds.jose.EncryptionMethod;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JOSEObject;
import com.nimbusds.jose.JWEAlgorithm;
import com.nimbusds.jose.JWEDecrypter;
import com.nimbusds.jose.JWEObject;
import com.nimbusds.jose.Payload;
import com.nimbusds.jose.crypto.AESDecrypter;
import com.nimbusds.jose.crypto.DirectDecrypter;
import com.nimbusds.jose.crypto.ECDHDecrypter;
import com.nimbusds.jose.crypto.RSADecrypter;
import com.nimbusds.jwt.EncryptedJWT;
import com.nimbusds.jwt.JWT;
import com.nimbusds.jwt.JWTParser;
import java.security.interfaces.ECPrivateKey;
import java.security.interfaces.RSAPrivateKey;
import java.text.ParseException;
import java.util.Collection;
import java.util.List;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.crypto.SecretKey;
import net.shibboleth.oidc.security.CredentialConversionUtil;
import net.shibboleth.oidc.security.credential.JWKCredential;
import net.shibboleth.oidc.security.credential.impl.DataEncryptionAlgorithmCriterion;
import net.shibboleth.oidc.security.credential.impl.EvaluableKeyIDCredentialCriterion;
import net.shibboleth.oidc.security.credential.impl.KeyManagmentAlgorithmCriterion;
import net.shibboleth.oidc.security.jose.DecryptionParameters;
import net.shibboleth.oidc.security.jose.criterion.JOSEObjectCriterion;
import net.shibboleth.oidc.security.jose.criterion.KeyIdCriterion;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.resolver.CriteriaSet;
import net.shibboleth.shared.resolver.Criterion;
import net.shibboleth.shared.resolver.ResolverException;
import org.opensaml.security.credential.Credential;
import org.opensaml.security.credential.CredentialResolver;
import org.opensaml.security.credential.UsageType;
import org.opensaml.security.criteria.KeyAlgorithmCriterion;
import org.opensaml.security.criteria.KeyLengthCriterion;
import org.opensaml.security.criteria.UsageCriterion;
import org.opensaml.xmlsec.algorithm.AlgorithmSupport;
import org.opensaml.xmlsec.encryption.support.DecryptionException;
import org.slf4j.Logger;

public class JWETokenDecrypter {
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(JWETokenDecrypter.class);
    @Nonnull
    private final DecryptionParameters params;

    public JWETokenDecrypter(@Nonnull DecryptionParameters decryptionParams) {
        this.params = (DecryptionParameters)Constraint.isNotNull((Object)decryptionParams, (String)"Decryption params can not be null");
    }

    @Nonnull
    public JWT decrypt(@Nonnull EncryptedJWT encryptedObject) throws DecryptionException {
        if (encryptedObject.getHeader() == null) {
            throw new DecryptionException("JWT headers are not available, decryption failed");
        }
        JWEAlgorithm jwtAlg = encryptedObject.getHeader().getAlgorithm();
        if (JWEAlgorithm.DIR.equals((Object)jwtAlg)) {
            this.decryptUsingDirectEncryption(encryptedObject);
        } else if (JWEAlgorithm.Family.RSA.contains((Object)jwtAlg)) {
            this.decryptUsingKeyEncryption(encryptedObject);
        } else if (JWEAlgorithm.Family.AES_GCM_KW.contains((Object)jwtAlg) || JWEAlgorithm.Family.AES_KW.contains((Object)jwtAlg)) {
            this.decryptUsingKeyWrapping(encryptedObject);
        } else if (JWEAlgorithm.Family.ECDH_ES.contains((Object)jwtAlg)) {
            this.decryptUsingKeyAgreement(encryptedObject);
        } else {
            throw new DecryptionException("JWE algorithm '" + jwtAlg.getName() + "' not supported");
        }
        Payload payload = encryptedObject.getPayload();
        if (payload != null && encryptedObject.getState() == JWEObject.State.DECRYPTED) {
            try {
                JWT parsedJwt = JWTParser.parse((String)payload.toString());
                assert (parsedJwt != null);
                return parsedJwt;
            }
            catch (ParseException e) {
                throw new DecryptionException("Error decrypting JWT", (Exception)e);
            }
        }
        throw new DecryptionException("JWE failed to decryptm, no error given");
    }

    @Nonnull
    private CriteriaSet buildCriteria(@Nonnull EncryptedJWT encryptedObject, @Nullable List<Criterion> criteria) {
        EncryptionMethod encMethod;
        JWEAlgorithm jweAlg;
        CriteriaSet newCriteriaSet = new CriteriaSet();
        if (!this.params.getAdditionalCriteria().isEmpty()) {
            this.params.getAdditionalCriteria().forEach(arg_0 -> ((CriteriaSet)newCriteriaSet).add(arg_0));
        }
        if (criteria != null) {
            criteria.forEach(arg_0 -> ((CriteriaSet)newCriteriaSet).add(arg_0));
        }
        if (encryptedObject.getHeader().getKeyID() != null) {
            String keyId = encryptedObject.getHeader().getKeyID();
            this.log.debug("Added keyID criteria: '{}'", (Object)keyId);
            newCriteriaSet.add((Object)new EvaluableKeyIDCredentialCriterion(new KeyIdCriterion(keyId)));
        }
        if ((jweAlg = encryptedObject.getHeader().getAlgorithm()) != null) {
            String algName = jweAlg.getName();
            assert (algName != null);
            newCriteriaSet.add((Object)new KeyManagmentAlgorithmCriterion(algName));
        }
        if ((encMethod = encryptedObject.getHeader().getEncryptionMethod()) != null) {
            String encMethodAlgName = encMethod.getName();
            assert (encMethodAlgName != null);
            newCriteriaSet.add((Object)new DataEncryptionAlgorithmCriterion(encMethodAlgName));
        }
        newCriteriaSet.add((Object)new JOSEObjectCriterion((JOSEObject)encryptedObject));
        return newCriteriaSet;
    }

    private void buildKeyManagementAlgorithmCriteria(@Nonnull CriteriaSet criteriaSet, @Nonnull EncryptedJWT encryptedObject) {
        KeyLengthCriterion lengthCrit;
        JWEAlgorithm alg = encryptedObject.getHeader().getAlgorithm();
        if (alg == null) {
            return;
        }
        String algAlgorithmURI = StringSupport.trimOrNull((String)alg.getName());
        if (algAlgorithmURI == null) {
            return;
        }
        KeyAlgorithmCriterion algoCrit = this.buildKeyAlgorithmCriteria(algAlgorithmURI);
        if (algoCrit != null) {
            criteriaSet.add((Object)algoCrit);
            this.log.debug("Added decryption key algorithm ('alg') criteria: {}", (Object)algoCrit.getKeyAlgorithm());
        }
        if ((lengthCrit = this.buildKeyLengthCriteria(algAlgorithmURI)) != null) {
            criteriaSet.add((Object)lengthCrit);
            this.log.debug("Added decryption key length criteria from EncryptionMethod algorithm URI: {}", (Object)lengthCrit.getKeyLength());
        }
    }

    private void buildContentEncryptionKeyAlgorithmCriteria(@Nonnull CriteriaSet criteriaSet, @Nonnull EncryptedJWT encryptedObject) {
        KeyLengthCriterion lengthCrit;
        EncryptionMethod enc = encryptedObject.getHeader().getEncryptionMethod();
        if (enc == null) {
            return;
        }
        String encAlgorithmURI = StringSupport.trimOrNull((String)enc.getName());
        if (encAlgorithmURI == null) {
            return;
        }
        KeyAlgorithmCriterion algoCrit = this.buildKeyAlgorithmCriteria(encAlgorithmURI);
        if (algoCrit != null) {
            criteriaSet.add((Object)algoCrit);
            this.log.debug("Added decryption key algorithm 'enc' criteria: {}", (Object)algoCrit.getKeyAlgorithm());
        }
        if ((lengthCrit = this.buildKeyLengthCriteria(encAlgorithmURI)) != null) {
            criteriaSet.add((Object)lengthCrit);
            this.log.debug("Added decryption key length criteria from EncryptionMethod algorithm URI: {}", (Object)lengthCrit.getKeyLength());
        } else if (enc.cekBitLength() != 0) {
            lengthCrit = new KeyLengthCriterion(enc.cekBitLength());
            criteriaSet.add((Object)lengthCrit);
            this.log.debug("Added decryption key length criteria from EncryptionMethod/KeySize: {}", (Object)lengthCrit.getKeyLength());
        }
    }

    @Nullable
    private KeyAlgorithmCriterion buildKeyAlgorithmCriteria(@Nullable String algorithmURI) {
        if (Strings.isNullOrEmpty((String)algorithmURI)) {
            return null;
        }
        assert (algorithmURI != null);
        String jcaKeyAlgorithm = AlgorithmSupport.getKeyAlgorithm((String)algorithmURI);
        if (!Strings.isNullOrEmpty((String)jcaKeyAlgorithm)) {
            assert (jcaKeyAlgorithm != null);
            return new KeyAlgorithmCriterion(jcaKeyAlgorithm);
        }
        return null;
    }

    @Nullable
    private KeyLengthCriterion buildKeyLengthCriteria(@Nullable String encAlgorithmURI) {
        if (Strings.isNullOrEmpty((String)encAlgorithmURI)) {
            return null;
        }
        assert (encAlgorithmURI != null);
        Integer keyLength = AlgorithmSupport.getKeyLength((String)encAlgorithmURI);
        if (keyLength != null) {
            return new KeyLengthCriterion(keyLength.intValue());
        }
        return null;
    }

    private void decryptUsingKeyAgreement(@Nonnull EncryptedJWT encryptedObject) throws DecryptionException {
        this.log.debug("Attempting decryption of JWE using Key Agreement managment mode");
        CredentialResolver kekResolver = this.params.getKEKCredentialResolver();
        if (kekResolver == null) {
            throw new DecryptionException("Decryption can not be attempted, KEK resolver is not available");
        }
        CriteriaSet criteria = this.buildCriteria(encryptedObject, CollectionSupport.singletonList((Object)new UsageCriterion(UsageType.ENCRYPTION)));
        this.buildKeyManagementAlgorithmCriteria(criteria, encryptedObject);
        try {
            for (Credential cred : kekResolver.resolve((Object)criteria)) {
                assert (cred != null);
                if (!(cred instanceof JWKCredential)) {
                    if (!this.log.isTraceEnabled()) continue;
                    this.log.trace("JWT decryption requires a JWK credential, resolved credential '{}' was not", (Object)CredentialConversionUtil.resolveKid((Credential)cred));
                    continue;
                }
                try {
                    this.validateKeyManagmentAlgorithm(encryptedObject, (JWKCredential)cred);
                    this.validateContentEncryptionAlgorithm(encryptedObject);
                    ECDHDecrypter decrypter = new ECDHDecrypter((ECPrivateKey)cred.getPrivateKey());
                    encryptedObject.decrypt((JWEDecrypter)decrypter);
                    return;
                }
                catch (JOSEException | DecryptionException e) {
                    this.log.debug("Failed to decrypt JWE using key '{}', continuing: {}", (Object)cred.getKeyNames(), (Object)e.getMessage());
                }
            }
        }
        catch (ResolverException e) {
            this.log.warn("Unable to decrypt JWE with Key Encryption", (Throwable)e);
        }
        throw new DecryptionException("All attempts to decrypt the JWE using a Key Wrapped CEK have failed");
    }

    private void decryptUsingKeyWrapping(@Nonnull EncryptedJWT encryptedObject) throws DecryptionException {
        this.log.debug("Attempting decryption of JWE using Key Wrapping managment mode");
        CredentialResolver kekResolver = this.params.getKEKCredentialResolver();
        if (kekResolver == null) {
            throw new DecryptionException("Decryption can not be attempted, KEK resolver is not available");
        }
        CriteriaSet criteria = this.buildCriteria(encryptedObject, CollectionSupport.singletonList((Object)new UsageCriterion(UsageType.ENCRYPTION)));
        this.buildKeyManagementAlgorithmCriteria(criteria, encryptedObject);
        try {
            for (Credential cred : kekResolver.resolve((Object)criteria)) {
                if (!(cred instanceof JWKCredential)) {
                    if (!this.log.isTraceEnabled()) continue;
                    this.log.trace("JWT decryption requires a JWK credential, resolved credential '{}' was not", (Object)CredentialConversionUtil.resolveKid((Credential)cred));
                    continue;
                }
                try {
                    this.validateKeyManagmentAlgorithm(encryptedObject, (JWKCredential)cred);
                    this.validateContentEncryptionAlgorithm(encryptedObject, (JWKCredential)cred);
                    AESDecrypter decrypter = new AESDecrypter(cred.getSecretKey());
                    encryptedObject.decrypt((JWEDecrypter)decrypter);
                    return;
                }
                catch (JOSEException | DecryptionException e) {
                    this.log.debug("Failed to decrypt JWE using key '{}', continuing: {}", (Object)cred.getKeyNames(), (Object)e.getMessage());
                }
            }
        }
        catch (ResolverException e) {
            this.log.warn("Unable to decrypt JWE with Key Encryption", (Throwable)e);
        }
        throw new DecryptionException("All attempts to decrypt the JWE using a Key Wrapped CEK have failed");
    }

    private void decryptUsingKeyEncryption(@Nonnull EncryptedJWT encryptedObject) throws DecryptionException {
        this.log.debug("Attempting decryption of JWE using Key Encryption managment mode");
        CredentialResolver kekResolver = this.params.getKEKCredentialResolver();
        if (kekResolver == null) {
            throw new DecryptionException("Decryption can not be attempted, KEK resolver is not available");
        }
        CriteriaSet criteria = this.buildCriteria(encryptedObject, CollectionSupport.singletonList((Object)new UsageCriterion(UsageType.ENCRYPTION)));
        this.buildKeyManagementAlgorithmCriteria(criteria, encryptedObject);
        try {
            for (Credential cred : kekResolver.resolve((Object)criteria)) {
                if (!(cred instanceof JWKCredential)) {
                    if (!this.log.isTraceEnabled()) continue;
                    this.log.trace("JWT decryption requires a JWK credential, resolved credential '{}' was not", (Object)CredentialConversionUtil.resolveKid((Credential)cred));
                    continue;
                }
                try {
                    this.validateKeyManagmentAlgorithm(encryptedObject, (JWKCredential)cred);
                    this.validateContentEncryptionAlgorithm(encryptedObject);
                    RSADecrypter decrypter = new RSADecrypter(cred.getPrivateKey());
                    encryptedObject.decrypt((JWEDecrypter)decrypter);
                    return;
                }
                catch (JOSEException | DecryptionException e) {
                    this.log.debug("Failed to decrypt JWE using key '{}', continuing: {}", (Object)cred.getKeyNames(), (Object)e.getMessage());
                }
            }
        }
        catch (ResolverException e) {
            this.log.warn("Unable to decrypt JWE with Key Encryption", (Throwable)e);
        }
        throw new DecryptionException("All attempts to decrypt the JWE using a Key Encrypted CEK have failed");
    }

    private void decryptUsingDirectEncryption(@Nonnull EncryptedJWT encryptedObject) throws DecryptionException {
        this.log.debug("Attempting decryption of JWE using Direct Encryption managment mode");
        CredentialResolver cekResolver = this.params.getContentEncryptionKeyCredentialResolver();
        if (cekResolver == null) {
            throw new DecryptionException("Decryption can not be attempted, CEK resolver is not available");
        }
        CriteriaSet criteria = this.buildCriteria(encryptedObject, List.of(new UsageCriterion(UsageType.ENCRYPTION)));
        this.buildContentEncryptionKeyAlgorithmCriteria(criteria, encryptedObject);
        try {
            for (Credential cred : cekResolver.resolve((Object)criteria)) {
                assert (cred != null);
                if (!(cred instanceof JWKCredential)) {
                    if (!this.log.isTraceEnabled()) continue;
                    this.log.trace("JWT decryption requires a JWK credential, resolved credential '{}' was not", (Object)CredentialConversionUtil.resolveKid((Credential)cred));
                    continue;
                }
                try {
                    this.validateKeyManagmentAlgorithm(encryptedObject, (JWKCredential)cred);
                    this.validateContentEncryptionAlgorithm(encryptedObject, (JWKCredential)cred);
                    DirectDecrypter decrypter = new DirectDecrypter(cred.getSecretKey());
                    encryptedObject.decrypt((JWEDecrypter)decrypter);
                    return;
                }
                catch (JOSEException | DecryptionException e) {
                    this.log.debug("Content encryption key '{}' failed to decrypt JWE, continuing: {}", (Object)cred.getKeyNames(), (Object)e.getMessage());
                }
            }
        }
        catch (ResolverException e) {
            this.log.warn("Unable to decrypt JWE using Direct Encryption", (Throwable)e);
        }
        throw new DecryptionException("Failed to decrypt JWE, no suitable credential found");
    }

    @Nonnull
    private JWEAlgorithm validateKeyManagmentAlgorithm(@Nonnull EncryptedJWT encryptedObject, @Nonnull JWKCredential cred) throws DecryptionException {
        SecretKey skey;
        if (encryptedObject.getHeader() == null) {
            throw new DecryptionException("JWE did not contain a JOSE header, is in an illegal state");
        }
        JWEAlgorithm headerAlg = encryptedObject.getHeader().getAlgorithm();
        this.validateAlgorithmURI(headerAlg.getName());
        if (cred.getAlgorithm() != null && !headerAlg.equals((Object)cred.getAlgorithm())) {
            throw new DecryptionException("Credential algorithm '" + String.valueOf(cred.getAlgorithm()) + "' was not a match for the algorithm '" + String.valueOf(encryptedObject.getHeader().getAlgorithm()) + "'");
        }
        if (JWEAlgorithm.Family.RSA.contains((Object)headerAlg) && !(cred.getPrivateKey() instanceof RSAPrivateKey)) {
            throw new DecryptionException("Credential did not contain an RSA private key");
        }
        if (JWEAlgorithm.Family.ECDH_ES.contains((Object)headerAlg) && !(cred.getPrivateKey() instanceof ECPrivateKey)) {
            throw new DecryptionException("Credential did not contain an EC private key");
        }
        if (!(!JWEAlgorithm.Family.AES_GCM_KW.contains((Object)headerAlg) && !JWEAlgorithm.Family.AES_KW.contains((Object)headerAlg) || (skey = cred.getSecretKey()) != null && "AES".equals(skey.getAlgorithm()))) {
            throw new DecryptionException("Credential did not contain an AES secret key");
        }
        if (JWEAlgorithm.DIR.equals((Object)headerAlg) && cred.getSecretKey() == null) {
            throw new DecryptionException("Credential did not contain a direct encryption secret key");
        }
        return headerAlg;
    }

    @Nonnull
    private EncryptionMethod validateContentEncryptionAlgorithm(@Nonnull EncryptedJWT encryptedObject) throws DecryptionException {
        if (encryptedObject.getHeader() == null) {
            throw new DecryptionException("JWE did not contain a JOSE header, is in an illegal state");
        }
        EncryptionMethod enc = encryptedObject.getHeader().getEncryptionMethod();
        if (enc == null) {
            throw new DecryptionException("JWE did not contain an 'enc' JOSE header, is in an illegal state");
        }
        this.validateAlgorithmURI(StringSupport.trimOrNull((String)enc.getName()));
        return enc;
    }

    @Nonnull
    private EncryptionMethod validateContentEncryptionAlgorithm(@Nonnull EncryptedJWT encryptedObject, @Nonnull JWKCredential cred) throws DecryptionException {
        String jcaKeyAlgorithm;
        SecretKey skey = cred.getSecretKey();
        if (skey == null) {
            throw new DecryptionException("Credential does not contain a content encryption secret key");
        }
        EncryptionMethod enc = this.validateContentEncryptionAlgorithm(encryptedObject);
        String encAlg = StringSupport.trimOrNull((String)enc.getName());
        String string = jcaKeyAlgorithm = encAlg != null ? AlgorithmSupport.getKeyAlgorithm((String)encAlg) : null;
        if (jcaKeyAlgorithm == null) {
            throw new DecryptionException("JOSE Header 'enc' algorithm is not supported by the algorithm registry");
        }
        if (!jcaKeyAlgorithm.equals(skey.getAlgorithm())) {
            throw new DecryptionException("JOSE Header 'enc' algorithm " + jcaKeyAlgorithm + " does not match credential algorithm " + skey.getAlgorithm());
        }
        return enc;
    }

    private void validateAlgorithmURI(@Nullable String algorithmURI) throws DecryptionException {
        if (algorithmURI == null) {
            throw new DecryptionException("Algorithm was null, failed include/exclude validation");
        }
        this.log.debug("Validating algorithm URI against include and exclude lists: algorithm: {}, included: {}, excluded: {}", new Object[]{algorithmURI, this.params.getIncludedAlgorithms(), this.params.getExcludedAlgorithms()});
        if (!AlgorithmSupport.validateAlgorithmURI((String)algorithmURI, (Collection)this.params.getIncludedAlgorithms(), (Collection)this.params.getExcludedAlgorithms())) {
            throw new DecryptionException("Algorithm failed include/exclude validation: " + algorithmURI);
        }
    }
}

