/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.shibboleth.idp.module.jetty;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.Map;

import javax.annotation.Nonnull;

import org.slf4j.Logger;

import net.shibboleth.idp.Version;
import net.shibboleth.idp.installer.impl.InstallerProperties;
import net.shibboleth.idp.module.impl.PluginIdPModule;
import net.shibboleth.profile.module.ModuleContext;
import net.shibboleth.profile.module.ModuleException;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.security.impl.SelfSignedCertificateGenerator;

public class JettyModule extends PluginIdPModule {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(JettyModule.class);

    /**
     * Constructor.
     */
    public JettyModule() throws IOException, ModuleException {
        super(Version.getVersion(), JettyModule.class);
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull
    public Map<ModuleResource, ResourceResult> enable(@Nonnull ModuleContext moduleContext) throws ModuleException {
        final Map<ModuleResource, ResourceResult> result = super.enable(moduleContext);
        final Path idpHome = Path.of(moduleContext.getInstallLocation());
        final Path keystore= idpHome.resolve("credentials").resolve("idp-userfacing.p12");
        if (!Files.exists(keystore)) {
            generateKeystore(keystore);
        }
        handleLogbackJars(idpHome, "jetty-base-12");
        return result;
    }

    /** Copy the logback jars into jetty-base\lib\logging and set up an appropriate ini file.
     * @param idpHome idpHome
     * @param jettyBaseName the jetty base to populate (parameterized to allow multiple jett-bases per plugin)
     * @throws ModuleException if we cannot find the logback version or get an IO issue.
     */
    private void handleLogbackJars(final @Nonnull Path idpHome, final @Nonnull @NotEmpty String jettyBaseName) throws ModuleException {

        final Path jettyBaseDir = idpHome.resolve(jettyBaseName);
        final Path jarTargetDir = jettyBaseDir.resolve("lib").resolve("logging");
        try {
            log.debug("Creating {}", jarTargetDir);
            Files.createDirectories(jarTargetDir);

            final Path distLibDir = idpHome.resolve("dist").resolve("webapp").resolve("WEB-INF").resolve("lib");
            final String version = inferLogbackVersion(distLibDir);

            copyLogbackJar(distLibDir, jarTargetDir, "logback-classic-" + version + ".jar");
            copyLogbackJar(distLibDir, jarTargetDir, "logback-core-" + version + ".jar");

            log.error("Writing logback ini");
            try(final PrintStream out = new PrintStream(jettyBaseDir.resolve("start.d").resolve("logback.ini").toFile())) {
                out.format("logback.version=%s\n", version);
            }
        }
        catch (final IOException ex) {
            log.error("Could not set up logback environment {}", ex);
            throw new ModuleException(ex);
        }
    }

    /** Copy a names logback jar from the IdP distribution to jetty-base 
     * @param sourceDir where to copy from (idp.home/dist/webapp/WEB-INF/lib
     * @param targetDir
     * @param jarName
     * @throws IOException if the copy fails
     */
    private void copyLogbackJar(Path sourceDir, Path targetDir, String jarName) throws IOException {

        final File targetFile = targetDir.resolve(jarName).toFile();

        if (targetFile.exists()) {
            log.debug("Logback jar {} already exists.", targetFile);
            return;
        }

        final File sourceFile = sourceDir.resolve(jarName).toFile();
        if (!sourceFile.exists()) {
            log.error("Logback jar {} does not exist.", sourceFile);
            throw new IOException("Logback jar not found");
        }

        log.debug("Copying {} to {}", sourceFile, targetFile);
        try(final InputStream in = new BufferedInputStream(new FileInputStream(sourceFile));
            final OutputStream out = new BufferedOutputStream(new FileOutputStream(targetFile))) {
              in.transferTo(out);
        }
    }

	/** Find out what version of logback this IdP install uses.
     * Look for logback-classic-version.jar and pluck out the version.
     * @param distLibDir {@link Path}for idp.home/dist/webapp/WEB-INF/lib
     * @return the version.
     * @throws ModuleException if the lookup failed.
     */
    private String inferLogbackVersion(Path distLibDir) throws ModuleException {

        try (DirectoryStream<Path> stream = Files.newDirectoryStream(distLibDir, "logback-classic-*.jar")) {
            Iterator<Path> i = stream.iterator();
            if (!i.hasNext()) {
                log.error("Could not locate logback-classic-*.jar in {}", distLibDir);
                throw new ModuleException("Could not locate logback jars ");
            }
            final String fileName = i.next().getFileName().toString();
            final int len = fileName.length();
            return fileName.substring(16, len-4);
        } catch (final IOException ex) {
            log.error("Error looking for logback-classic-*.jar in {}", distLibDir, ex);
            throw new ModuleException(ex);
        }
    }

	/** Generate an mock idp-userfacing.p12 keystore.
     * @param keystore where to put it
     * @throws ModuleException if the generator fails
     */
    private void generateKeystore(Path keyStore) throws ModuleException {
        final SelfSignedCertificateGenerator generator = new SelfSignedCertificateGenerator();
        generator.setKeystoreFile(keyStore.toFile());
        generator.setKeySize(InstallerProperties.DEFAULT_KEY_SIZE);
        final String hostName = "localhost";
        generator.setHostName(hostName);
        final String altName = "https://" + hostName + "/idp/shibboleth";
        generator.setURISubjectAltNames(CollectionSupport.singletonList(altName));
        generator.setKeystorePassword("changeit");
        log.info("Creating {}, CN = {} URI = {}, keySize={}",
                keyStore, hostName, altName, InstallerProperties.DEFAULT_KEY_SIZE);
        try {
          generator.generate();
      } catch (final Exception e) {
          log.error("Error building ketstore files {}", keyStore, e);
          throw new ModuleException("Error Key Store", e);
      }
    }
}
